<?php
session_start();
require_once __DIR__ . '/../includes/db.php';
    // At the top of your booking page
require_once __DIR__ . '/../includes/mail_functions.php';
require 'get_rooms.php';


function generateAlphanumeric($length = 100) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    
    for ($i = 0; $i < $length; $i++) {
        $randomChar = $characters[rand(0, $charactersLength - 1)];
        $randomString .= $randomChar;
    }
    
    return $randomString;
}

// Example usage
//echo generateAlphanumeric(12); // Outputs something like 'a8F9kLm3Xz0T'






// Fetch categories
if (!isset($_SESSION['user'])) {
    header("Location: ../index.php");
    exit;
}
$admin = $_SESSION['user'];
$categories = $pdo->query("SELECT * FROM room_categories")->fetchAll(PDO::FETCH_ASSOC);

// Handle form submission
$message = '';
$error = '';

if ($_SERVER["REQUEST_METHOD"] === "POST") {
    $name      = trim($_POST['name']);
    $phone     = trim($_POST['phone']);
    $email     = trim($_POST['email']);
    $checkin   = $_POST['checkin_date'];
    $checkout  = $_POST['checkout_date'];
    $category  = $_POST['room_category'];
    $room_id   = $_POST['room_id'];
    $booking_type = $POST['bookingType'];
    $price_raw = preg_replace('/[^0-9.]/', '', $_POST['price']); // Remove NGN symbols
    $admin_id  = $admin['id']; // assuming admin is logged in
    //'pending','confirmed','cancelled','checked_in','checked_out'
    $booking_ID ="@sheridan_".generateAlphanumeric(7)

    if (empty($name) || empty($phone) || empty($email) || empty($checkin) || empty($checkout) || empty($category) || empty($room_id)) {
        $error = "All fields are required.";
    } elseif (strtotime($checkin) >= strtotime($checkout)) {
        $error = "Check-out date must be after check-in date.";
    } else {
        $roomCheck = $pdo->prepare("SELECT status FROM rooms WHERE id = ?");
        $roomCheck->execute([$room_id]);
        $roomStatus = $roomCheck->fetchColumn();

        if ($roomStatus !== 'available') {
            $error = "This room has already been booked. Please choose another.";
        } else {
            // Calculate number of nights
            $checkinDate = new DateTime($checkin);
            $checkoutDate = new DateTime($checkout);
            $nights = $checkinDate->diff($checkoutDate)->days;

            $totalAmount = $nights * floatval($price_raw);

            try {
                $pdo->beginTransaction();

                // Insert customer
                $stmtCustomer = $pdo->prepare("INSERT INTO customers (full_name, email, phone, address, created_at) VALUES (?, ?, ?, '', NOW())");
                $stmtCustomer->execute([$name, $email, $phone]);
                $customer_id = $pdo->lastInsertId();

                // Insert booking
                $stmtBooking = $pdo->prepare("INSERT INTO bookings (`customer_id`, `booked_by`, `room_id`, `check_in`, `check_out`, `total_amount`, `status`, `created_at`, `booking_type`,`booking_ID`) VALUES (?, ?, ?, ?, ?, ?, 'booked', NOW(), ?)");
                $stmtBooking->execute([$customer_id, $admin_id, $room_id, $checkin, $checkout, $totalAmount,$bookingType,$booking_ID]);

                // Update room status
                $updateRoomStat=$pdo->prepare("UPDATE rooms SET `status` = 'occupied' WHERE id = ?");
                $updateRoomStat->execute([$room_id]);

                $pdo->commit();
                $message = "Room successfully booked!";
                
                // After successful booking
                $booking_details = [
                    'booking_id' => $booking_ID,
                    'guest_name' => $name,
                    'room_type' => $category,
                    'check_in' => $checkin,
                    'check_out' => $checkout,
                    'total_amount' => $totalAmount
                ];

                $email_sent = sendEmail([
                    'to_email' => $email,
                    'to_name' => $name,
                    'subject' => 'Your Sheridan Hotel Booking Confirmation',
                    'body' => generateBookingEmail($booking_details)
                ]);

                if (!$email_sent) {
                    // Log error or notify admin
                    error_log("Failed to send booking confirmation to $customerEmail");
                }





            } catch (Exception $e) {
                $pdo->rollBack();
                $error = "Booking failed: " . $e->getMessage();
            }
        }
    }

}


?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Book a Room</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 flex h-screen">
    <aside class="w-64 bg-white shadow-md h-full flex flex-col">
        <div class="p-6">
            <h1 class="text-xl font-bold text-blue-700">Sheridan Hotel</h1>
            <p class="text-gray-600 mb-6">You are logged in as <strong><?= $admin['role'] ?></strong>.</p>
        </div>
        <nav class="flex-1 px-4 space-y-2">
            <a href="dashboard.php" class="block px-4 py-2 text-gray-700 rounded hover:bg-gray-200">Dashboard Home</a>
            <a href="book_room.php" class="block px-4 py-2 bg-blue-100 text-blue-800 rounded">Book a Room</a>
            <a href="room_status.php" class="block px-4 py-2 text-gray-700 rounded hover:bg-gray-200">Check Room Status</a>
            <a href="available_rooms.php" class="block px-4 py-2 text-gray-700 rounded hover:bg-gray-200">Available Rooms</a>
            <a href="check_out.php" class="block px-4 py-2 text-gray-700 rounded hover:bg-gray-200">Customer Check-out</a>
            <a href="list_booked.php" class="block px-4 py-2 text-gray-700 rounded hover:bg-gray-200">List of Booked Rooms</a>
        </nav>
        <div class="p-4 border-t">
            <p class="text-sm text-gray-500">All right reserved &copy;Sheridan Hotels and suits</p>
            <a href="../logout.php" class="mt-2 inline-block text-red-500 hover:underline text-sm">Logout</a>
        </div>
    </aside>

    <div class="flex-1 flex flex-col overflow-hidden">
        <nav class="bg-white shadow px-6 py-4 flex justify-between items-center">
            <h1 class="text-xl font-bold text-gray-800">Dashboard Overview</h1>
            <div class="flex items-center gap-4">
                <span class="text-gray-600 text-sm">Welcome, <?= htmlspecialchars($admin['name']) ?>  <span id="datetime"></span></span>
            </div>
        </nav>

        <div class="max-w-2xl mx-auto bg-white p-6 rounded shadow">
            <h2 class="text-xl font-bold mb-4">Book a Room</h2>

            <?php if ($message): ?>
                <div class="bg-green-100 text-green-700 p-2 rounded mb-4"> <?= htmlspecialchars($message) ?> </div>
            <?php elseif ($error): ?>
                <div class="bg-red-100 text-red-700 p-2 rounded mb-4"> <?= htmlspecialchars($error) ?> </div>
            <?php endif; ?>

            <form method="POST">
                <input type="text" name="name" placeholder="Guest Name" required class="w-full mb-3 p-2 border rounded">
                <input type="text" name="phone" placeholder="Phone Number" required class="w-full mb-3 p-2 border rounded">
                <input type="email" name="email" placeholder="Email Address" required class="w-full mb-3 p-2 border rounded">

                <label class="block mt-2 mb-1">Room Category:</label>
                <select name="room_category" id="room_category" required class="w-full mb-3 p-2 border rounded">
                    <option value="">Select Category</option>
                    <?php foreach ($categories as $cat): ?>
                        <option value="<?= $cat['id'] ?>"> <?= htmlspecialchars($cat['name']) ?> </option>
                    <?php endforeach; ?>
                </select>

                <label class="block mt-2 mb-1">Available Rooms:</label>
                <select name="room_id" id="room_id" required class="w-full mb-3 p-2 border rounded">
                    <option value="">Select a category first</option>
                </select>

                <label class="block mt-2 mb-1">Check-in Date:</label>
                <input type="date" name="checkin_date" required class="w-full mb-3 p-2 border rounded">

                <label class="block mt-2 mb-1">Check-out Date:</label>
                <input type="date" name="checkout_date" required class="w-full mb-3 p-2 border rounded">

                <label class="block mt-2 mb-1">Room Price:</label>
                <input type="text" name="price" id="price" readonly class="w-full mb-3 p-2 border rounded bg-gray-100">
                
                <label class="block mt-2 mb-1">Booking Type:</label>
                <select name="bookingType" id="bookingType" required class="w-full mb-3 p-2 border rounded">
                    <option value="">Select a category first</option>
                    <option value="online">Online Booking(Via Mail)</option>
                    <option value="walk-in">Walking</option>
                    <option value="phone">Phone Call Or Whatsapp</option>
                </select>


                <button class="w-full bg-blue-600 text-white p-2 rounded hover:bg-blue-700" type="submit">Book Room</button>
            </form>
        </div>
    </div>

    <script>
                function formatCurrency(amount) {
                    return new Intl.NumberFormat('en-NG', {
                    style: 'currency',
                    currency: 'NGN',
                    }).format(amount);
                    }

        // Set up datetime display
        setInterval(() => {
            document.getElementById('datetime').textContent = new Date().toLocaleString(undefined, { hour12: false });
        }, 1000);

        let currentRoomData = [];

        async function loadRooms() {
            const category = document.getElementById('room_category').value;
            const roomSelect = document.getElementById('room_id');
            const priceInput = document.getElementById('price');
            
            // Clear previous data
            roomSelect.innerHTML = '<option value="">Loading rooms...</option>';
            priceInput.value = '';
            
            if (!category) {
                roomSelect.innerHTML = '<option value="">Select a category first</option>';
                return;
            }

            try {
                const res = await fetch(`get_rooms.php?category_id=${category}`);
                
                if (!res.ok) {
                    throw new Error(`HTTP error! status: ${res.status}`);
                }
                
                const data = await res.json();
                currentRoomData = data;

                roomSelect.innerHTML = '';

                if (data.length === 0) {
                    roomSelect.innerHTML = '<option value="">No rooms available</option>';
                    priceInput.value = '';
                    return;
                }

                data.forEach(room => {
                    const option = document.createElement('option');
                    option.value = room.id;
                    option.textContent = `Room ${room.room_number}`;
                    roomSelect.appendChild(option);
                });

                // Set price for the first room (default)
                if (data.length > 0) {
                    priceInput.value = data[0].price_per_night;
                }
            } catch (error) {
                console.error('Error loading rooms:', error);
                roomSelect.innerHTML = '<option value="">Error loading rooms</option>';
                priceInput.value = '';
            }
        }

        // Set up event listeners when DOM is loaded
        document.addEventListener('DOMContentLoaded', () => {
            const roomCategorySelect = document.getElementById('room_category');
            const roomSelect = document.getElementById('room_id');
            const priceInput = document.getElementById('price');

            // Add change event listener to room category
            roomCategorySelect.addEventListener('change', loadRooms);

            // Update price when room selection changes
            roomSelect.addEventListener('change', () => {
                const selectedRoomId = roomSelect.value;
                if (selectedRoomId && currentRoomData.length > 0) {
                    const selectedRoom = currentRoomData.find(r => r.id == selectedRoomId);
                    priceInput.value = selectedRoom ? formatCurrency(selectedRoom.price_per_night) : '';
                } else {
                    priceInput.value = '';
                }
            });
        });
    </script>
</body>
</html>