<?php
require_once __DIR__ . '/../includes/db.php';
session_start();

if (!isset($_SESSION['user'])) {
    header("Location:./index.php");
    exit;
}

$admin = $_SESSION['user'];

// --- Pagination Logic ---
$records_per_page = 20; // Number of records to display per page

// Get current page number from URL, default to 1
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
if ($current_page < 1) {
    $current_page = 1;
}

// Calculate the OFFSET for the SQL query
$offset = ($current_page - 1) * $records_per_page;

// Fetch total number of records for pagination calculation
$total_records_stmt = $pdo->query("SELECT COUNT(*) FROM bookings");
$total_records = $total_records_stmt->fetchColumn();

// Calculate total pages
$total_pages = ceil($total_records / $records_per_page);

// Fetch all booked rooms, including the booking_ID and apply LIMIT/OFFSET
$stmt = $pdo->prepare("SELECT b.id, b.booking_ID, b.check_in, b.check_out, b.status,
                            r.room_number, r.price_per_night,
                            c.name AS category,
                            cust.full_name AS guest_name
                     FROM bookings b
                     JOIN rooms r ON b.room_id = r.id
                     JOIN room_categories c ON r.category_id = c.id
                     JOIN customers cust ON b.customer_id = cust.id
                     ORDER BY b.check_in DESC
                     LIMIT :limit OFFSET :offset");
$stmt->bindParam(':limit', $records_per_page, PDO::PARAM_INT);
$stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();

$bookedRooms = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Booked Rooms</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        body {
            font-family: 'Inter', sans-serif;
        }
        
        .gradient-bg {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }
        
        .nav-item {
            transition: all 0.2s ease;
            position: relative;
            overflow: hidden;
        }
        
        .nav-item:hover {
            background: linear-gradient(135deg, #f3f4f6, #e5e7eb);
            transform: translateX(5px);
        }
        
        .nav-item.active {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
        }
        
        .metric-card {
            background: linear-gradient(135deg, #ffffff, #f8fafc);
            border: 1px solid rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
        }
        
        .sidebar-gradient {
            background: linear-gradient(180deg, #1e293b 0%, #334155 100%);
        }
        
        .animate-fade-in {
            animation: fadeIn 0.8s ease-in-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .pulse-dot {
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 1; }
            50% { transform: scale(1.1); opacity: 0.8; }
        }

        /* Enhanced Table Styles */
        .styled-table {
            border-collapse: separate; /* Use separate to allow border-radius */
            border-spacing: 0; /* Remove space between cells */
            overflow: hidden; /* Ensures rounded corners are visible */
        }
        .styled-table thead th {
            padding: 1rem 1.5rem;
            text-align: left;
            font-weight: 600;
            background: linear-gradient(to right, #4F46E5, #6366F1); /* Stronger gradient */
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        .styled-table tbody td {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid #E5E7EB; /* Lighter border for rows */
            color: #4B5563; /* Darker text for readability */
        }
        .styled-table tbody tr:last-child td {
            border-bottom: none; /* No border on the last row */
        }
        .styled-table tbody tr:nth-child(odd) {
            background-color: #F9FAFB; /* Zebra striping */
        }
        .styled-table tbody tr:hover {
            background-color: #EEF2FF; /* Lighter blue on hover */
            cursor: pointer;
            transform: scale(1.005); /* Slight scale for hover effect */
            transition: background-color 0.2s ease-in-out, transform 0.1s ease-out;
        }
        .status-booked {
            background-color: #D1FAE5; /* Light green */
            color: #065F46; /* Dark green text */
            font-weight: 500;
            padding: 0.25rem 0.75rem;
            border-radius: 9999px; /* Fully rounded */
        }
        .status-checked-in {
            background-color: #BFDBFE; /* Light blue */
            color: #1D4ED8; /* Dark blue text */
            font-weight: 500;
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
        }
        .status-other {
            background-color: #E5E7EB; /* Light gray */
            color: #4B5563; /* Gray text */
            font-weight: 500;
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
        }
        .action-button {
            background: linear-gradient(to right, #8B5CF6, #A78BFA); /* Purple gradient */
            color: white;
            font-weight: 500;
            padding: 0.5rem 1rem;
            border-radius: 0.5rem;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            transition: all 0.2s ease-in-out;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        .action-button:hover {
            background: linear-gradient(to right, #7C3AED, #8B5CF6);
            transform: translateY(-2px);
            box-shadow: 0 6px 8px rgba(0, 0, 0, 0.15);
        }

        /* Pagination styles */
        .pagination-container {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 1.5rem;
            gap: 0.5rem;
        }
        .pagination-button {
            padding: 0.5rem 1rem;
            border-radius: 0.5rem;
            background-color: #E0E7FF; /* Light blue background */
            color: #4F46E5; /* Indigo text */
            font-weight: 600;
            transition: all 0.2s ease;
        }
        .pagination-button:hover:not(.disabled):not(.active) {
            background-color: #C7D2FE; /* Lighter blue on hover */
            transform: translateY(-1px);
        }
        .pagination-button.active {
            background-color: #4F46E5; /* Dark blue when active */
            color: white;
            cursor: default;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .pagination-button.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
    </style>
</head>
<body class="bg-gradient-to-br from-gray-50 to-gray-100 min-h-screen">
<!-- Sidebar -->
    <aside class="fixed left-0 top-0 w-72 h-full sidebar-gradient shadow-2xl z-50">
        <div class="p-8">
            <div class="flex items-center gap-3 mb-2">
                <div class="w-10 h-10 bg-gradient-to-r from-blue-400 to-purple-500 rounded-xl flex items-center justify-center">
                    <i class="fas fa-hotel text-white text-lg"></i>
                </div>
                <h1 class="text-2xl font-bold text-white">Sheridan Hotel</h1>
            </div>
            <div class="flex items-center gap-2 mt-4 p-3 bg-white/10 rounded-lg backdrop-blur-sm">
                <div class="w-2 h-2 bg-green-400 rounded-full pulse-dot"></div>
                <p class="text-white/90 text-sm">Logged in as <span class="font-semibold text-blue-300"><?= $admin['role'] ?></span></p>
            </div>
        </div>
        
        <nav class="px-6 space-y-2">
            <a href="dashboard.php" class="nav-item flex items-center gap-3 px-4 py-3 text-white/80 hover:text-white rounded-xl">
                <i class="fas fa-tachometer-alt w-5"></i>
                <span class="font-medium">Dashboard Home</span>
            </a>
            <a href="book_room.php" class="nav-item flex items-center gap-3 px-4 py-3 text-white/80 hover:text-white rounded-xl">
                <i class="fas fa-calendar-plus w-5"></i>
                <span class="font-medium">Book a Room</span>
            </a>
            <a href="room_status.php" class="nav-item flex items-center gap-3 px-4 py-3 text-white/80 hover:text-white rounded-xl">
                <i class="fas fa-bed w-5"></i>
                <span class="font-medium">Room Status</span>
            </a>
            <a href="check_out.php" class="nav-item flex items-center gap-3 px-4 py-3 text-white/80 hover:text-white rounded-xl">
                <i class="fas fa-sign-out-alt w-5"></i>
                <span class="font-medium">Customer Check-out</span>
            </a>
            <a href="list_booked.php" class="nav-item active flex items-center gap-3 px-4 py-3 text-white rounded-xl">
                <i class="fas fa-list-alt w-5"></i>
                <span class="font-medium">Booked Rooms</span>
            </a>
            <a href="list_all_rooms.php" class="nav-item flex items-center gap-3 px-4 py-3 text-white/80 hover:text-white rounded-xl">
                <i class="fas fa-building w-5"></i>
                <span class="font-medium">All Rooms</span>
            </a>
             <a href="addstaff.php" class="nav-item flex items-center gap-3 px-4 py-3 text-white/80 hover:text-white rounded-xl">
                <i class="fas fa-user-plus w-5"></i>
                <span class="font-medium">Add Staff</span>
            </a>
        </nav>
        
        <div class="absolute bottom-0 left-0 right-0 p-6 border-t border-white/10">
            <p class="text-white/60 text-xs mb-3">© Sheridan Hotels and Suites</p>
            <a href="../logout.php" class="flex items-center gap-2 text-red-400 hover:text-red-300 text-sm font-medium transition-colors">
                <i class="fas fa-power-off"></i>
                <span>Logout</span>
            </a>
        </div>
    </aside>

<div class="ml-72 min-h-screen">
<div class="flex-1 flex flex-col overflow-hidden">
    <header class="bg-white/80 backdrop-blur-sm border-b border-gray-200/50 px-8 py-6 sticky top-0 z-40">
            <div class="flex justify-between items-center">
                <div>
                    <h1 class="text-3xl font-bold text-gray-800">Booked Rooms</h1>
                    <p class="text-gray-600 mt-1">List of all currently booked rooms</p>
                </div>
                <div class="flex items-center gap-6">
                    <div class="text-right">
                        <p class="text-sm text-gray-500">Welcome back,</p>
                        <p class="font-semibold text-gray-800"><a href="My_profile.php"><?= htmlspecialchars($admin['name']) ?></a></p>
                    </div>
                    <div class="w-12 h-12 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white font-bold text-lg">
                        <?= strtoupper(substr($admin['name'], 0, 1)) ?>
                    </div>
                    <div class="text-sm text-gray-600 font-mono bg-gray-100 px-3 py-2 rounded-lg">
                        <span id="datetime"></span>
                    </div>
                </div>
            </div>
        </header>

<main class="p-8">
    
    <div class="p-6 overflow-x-auto"> <!-- Use overflow-x-auto for responsiveness -->
        <?php if (count($bookedRooms) > 0): ?>
            <table class="min-w-full bg-white shadow-xl rounded-lg styled-table"> <!-- Added styled-table class -->
                <thead class="">
                    <tr>
                        <th class="">Room Number</th>
                        <th class="">Category</th>
                        <th class="">Guest Name</th>
                        <th class="">Check-In</th>
                        <th class="">Check-Out</th>
                        <th class="">Status</th>
                        <th class="">Actions</th> 
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($bookedRooms as $booking): 
                        $statusClass = 'status-other';
                        if ($booking['status'] == 'booked') {
                            $statusClass = 'status-booked';
                        } elseif ($booking['status'] == 'checked_in') {
                            $statusClass = 'status-checked-in';
                        }
                    ?>
                        <tr>
                            <td><?= htmlspecialchars($booking['room_number']) ?></td>
                            <td><?= htmlspecialchars($booking['category']) ?></td>
                            <td><?= htmlspecialchars($booking['guest_name']) ?></td>
                            <td><?= htmlspecialchars($booking['check_in']) ?></td>
                            <td><?= htmlspecialchars($booking['check_out']) ?></td>
                            <td>
                                <span class="<?= $statusClass ?>">
                                    <?= ucfirst(str_replace('_', ' ', $booking['status'])) ?>
                                </span>
                            </td>
                            <td>
                                <?php if (!empty($booking['booking_ID'])): ?>
                                    <button onclick="openReceiptModal('<?= htmlspecialchars($booking['booking_ID']) ?>')" 
                                            class="action-button">
                                        <i class="fas fa-receipt"></i> View
                                    </button>
                                <?php else: ?>
                                    <span class="text-gray-500 text-xs">N/A</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <!-- Pagination Controls -->
            <div class="pagination-container">
                <?php if ($current_page > 1): ?>
                    <a href="?page=<?= $current_page - 1 ?>" class="pagination-button">Previous</a>
                <?php else: ?>
                    <span class="pagination-button disabled">Previous</span>
                <?php endif; ?>

                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                    <a href="?page=<?= $i ?>" class="pagination-button <?= ($i == $current_page) ? 'active' : '' ?>">
                        <?= $i ?>
                    </a>
                <?php endfor; ?>

                <?php if ($current_page < $total_pages): ?>
                    <a href="?page=<?= $current_page + 1 ?>" class="pagination-button">Next</a>
                <?php else: ?>
                    <span class="pagination-button disabled">Next</span>
                <?php endif; ?>
            </div>

        <?php else: ?>
            <div class="bg-yellow-100 text-yellow-800 p-4 rounded-lg shadow-md">No rooms are currently booked.</div>
        <?php endif; ?>
    </div>
        </main>
</div>

<!-- Receipt Modal - Add this HTML block at the very end of your body tag -->
<div id="receiptModal" class="fixed inset-0 bg-gray-900 bg-opacity-75 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg shadow-xl overflow-hidden max-w-2xl w-full mx-4" style="max-height: 90vh;">
        <!-- Modal Header -->
        <div class="bg-gradient-to-r from-blue-600 to-purple-600 p-5 text-white flex justify-between items-center">
            <h3 class="text-xl font-semibold">Booking Receipt</h3>
            <button id="closeReceiptModal" class="text-white hover:text-gray-200">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>
        <!-- Modal Body - where the receipt content goes, loaded in an iframe -->
        <div class="p-4 overflow-y-auto" style="max-height: calc(90vh - 120px);"> <!-- Adjust height for header/footer -->
            <iframe id="receiptIframe" src="" frameborder="0" class="w-full h-full min-h-[400px]"></iframe>
        </div>
        <!-- Modal Footer -->
        <div class="p-4 border-t border-gray-200 flex justify-end gap-3">
            <button id="printReceiptThermal" class="bg-blue-500 hover:bg-blue-600 text-white font-bold py-2 px-4 rounded-lg flex items-center gap-2 shadow-md transition-transform transform hover:scale-105">
                <i class="fas fa-print"></i> Print Receipt
            </button>
        </div>
    </div>
</div>

<script>
    setInterval(() => {
            document.getElementById('datetime').textContent = new Date().toLocaleString(undefined, { 
                hour12: false,
                year: 'numeric',
                month: 'short',
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
        }, 1000);

    // --- Modal Specific JavaScript ---
    const receiptModal = document.getElementById('receiptModal');
    const closeReceiptModalBtn = document.getElementById('closeReceiptModal');
    const receiptIframe = document.getElementById('receiptIframe');
    const printReceiptThermalBtn = document.getElementById('printReceiptThermal');

    // Function to open the receipt modal (made global for easy access from button clicks)
    window.openReceiptModal = function(bookingId) {
        if (bookingId) {
            // Load the content of receipt_thermal.php into the iframe
            // Ensure the path to receipt_thermal.php is correct relative to list_booked.php
            receiptIframe.src = `receipt_thermal.php?booking_ID=${encodeURIComponent(bookingId)}`;
            receiptModal.classList.remove('hidden'); // Show the modal by removing 'hidden' class
        } else {
            console.error("No booking ID provided to open receipt modal.");
        }
    };

    // Event listener to close the modal
    closeReceiptModalBtn.addEventListener('click', () => {
        receiptModal.classList.add('hidden'); // Hide the modal
        receiptIframe.src = ''; // Clear iframe content to prevent re-display on next open
    });

    // Event listener for printing the receipt from the iframe
    printReceiptThermalBtn.addEventListener('click', () => {
        if (receiptIframe.contentWindow) {
            receiptIframe.contentWindow.print(); // Trigger the browser's print dialog for the iframe content
        } else {
            console.error("Could not access iframe contentWindow for printing.");
        }
    });

    // Optional: Close modal if user clicks outside of the modal content
    receiptModal.addEventListener('click', (e) => {
        if (e.target === receiptModal) {
            receiptModal.classList.add('hidden');
            receiptIframe.src = '';
        }
    });
</script>
</body>
</html>