<?php
session_start();
require_once __DIR__ . '/../includes/db.php';

// Ensure user is logged in for security
if (!isset($_SESSION['user'])) {
    // For a modal context, you might not redirect, but rather just output an error message
    // or log it, as the user won't directly navigate here.
    exit("Unauthorized access to receipt generation.");
}

$booking_ID = $_GET['booking_ID'] ?? ''; // Get booking ID from URL parameter

$receipt_details = null;
if ($booking_ID) {
    // Fetch all necessary booking, customer, room, and category details
    $stmt = $pdo->prepare("
        SELECT 
            b.booking_ID, 
            c.full_name, c.email, c.phone, c.address,
            r.room_number,
            rc.name as room_category_name,
            rc.price_per_night, -- Fetch price per night from room_categories for detailed breakdown
            b.check_in, b.check_out, b.total_amount, b.created_at
        FROM bookings b
        JOIN customers c ON b.customer_id = c.id
        JOIN rooms r ON b.room_id = r.id
        JOIN room_categories rc ON r.category_id = rc.id
        WHERE b.booking_ID = ?
    ");
    $stmt->execute([$booking_ID]);
    $receipt_details = $stmt->fetch(PDO::FETCH_ASSOC);
}

// If no receipt details are found, display an error and exit
if (!$receipt_details) {
    exit("Receipt not found or invalid booking ID provided.");
}

// Calculate number of nights for display on the receipt
$checkinDate = new DateTime($receipt_details['check_in']);
$checkoutDate = new DateTime($receipt_details['check_out']);
$nights = $checkinDate->diff($checkoutDate)->days;
if ($nights == 0) {
    $nights = 1; // If check-in and check-out are on the same day, count as 1 night
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sheridan Hotel Receipt</title>
    <style>
        /* Define styles specifically for thermal printer output */
        body {
            width: 80mm; /* Standard thermal printer paper width (adjust as needed, e.g., 58mm) */
            margin: 0;
            padding: 5mm; /* Small padding around the content */
            font-family: 'monospace', sans-serif; /* Use a monospace font for crisp, clear text */
            font-size: 10px; /* Adjust font size for readability on small receipts */
            color: #000; /* Ensure text is black */
        }
        .center { text-align: center; }
        .right { text-align: right; }
        .line { border-top: 1px dashed #000; margin: 5px 0; } /* Dashed line for separation */
        .bold { font-weight: bold; }
        h1, h2, h3, h4, h5, h6 { margin: 0; padding: 0; } /* Remove default margins for headings */
        
        /* Flexbox for detail rows to align key-value pairs */
        .details-row { 
            display: flex; 
            justify-content: space-between; 
            margin-bottom: 2px; 
            white-space: nowrap; /* Prevent wrapping for small lines */
        }
        .total { 
            font-size: 14px; 
            font-weight: bold; 
            margin-top: 10px; 
            text-align: center;
        }

        /* Styles specifically for print media to hide non-receipt elements */
        @media print {
            body { 
                margin: 0 !important; 
                padding: 0 !important; 
                -webkit-print-color-adjust: exact; /* Ensure background colors/images print */
                print-color-adjust: exact;
            }
            .no-print { display: none; } /* Hide elements with this class when printing */
        }
    </style>
</head>
<body>
    <div class="center">
        <h2 class="bold">SHERIDAN HOTEL</h2>
        <p>123 Hotel Main Street, Akure, Ondo State</p>
        <p>+234 801 234 5678 | info@sheridanhotel.com</p>
    </div>

    <div class="line"></div>

    <div class="details-row">
        <span>RECEIPT ID:</span>
        <span class="bold"><?= htmlspecialchars($receipt_details['booking_ID']) ?></span>
    </div>
    <div class="details-row">
        <span>DATE ISSUED:</span>
        <span><?= date('Y-m-d H:i') ?></span>
    </div>
    <div class="details-row">
        <span>BOOKING DATE:</span>
        <span><?= date('Y-m-d H:i', strtotime($receipt_details['created_at'])) ?></span>
    </div>

    <div class="line"></div>

    <div class="details-row">
        <span class="bold">GUEST NAME:</span>
        <span class="bold"><?= htmlspecialchars(strtoupper($receipt_details['full_name'])) ?></span>
    </div>
    <div class="details-row">
        <span>EMAIL:</span>
        <span><?= htmlspecialchars($receipt_details['email']) ?></span>
    </div>
    <div class="details-row">
        <span>PHONE:</span>
        <span><?= htmlspecialchars($receipt_details['phone']) ?></span>
    </div>

    <div class="line"></div>

    <div class="details-row">
        <span>ROOM:</span>
        <span class="bold"><?= htmlspecialchars($receipt_details['room_number']) ?> (<?= htmlspecialchars($receipt_details['room_category_name']) ?>)</span>
    </div>
    <div class="details-row">
        <span>CHECK-IN:</span>
        <span><?= htmlspecialchars($receipt_details['check_in']) ?></span>
    </div>
    <div class="details-row">
        <span>CHECK-OUT:</span>
        <span><?= htmlspecialchars($receipt_details['check_out']) ?></span>
    </div>
    <div class="details-row">
        <span>NUMBER OF NIGHTS:</span>
        <span><?= $nights ?></span>
    </div>
    <div class="details-row">
        <span>PRICE PER NIGHT:</span>
        <span class="right"><?= number_format($receipt_details['price_per_night'], 2) ?> NGN</span>
    </div>

    <div class="line"></div>

    <div class="total center">
        TOTAL AMOUNT PAID
        <br>
        <span style="font-size: 20px;"><?= number_format($receipt_details['total_amount'], 2) ?> NGN</span>
    </div>

    <div class="line"></div>

    <div class="center">
        <p>THANK YOU FOR YOUR STAY!</p>
        <p>We hope to see you again soon.</p>
        <p style="margin-top: 10px;">Powered by Sheridan PMS</p>
    </div>

</body>
</html>